<?php
/*
Plugin Name: TagLite 推薦文章插件
Description: 根據 TagLite 使用者標籤，自動抓取並顯示推薦文章/商品（支援 WooCommerce），可自訂樣式與模板；無標籤時隨機推薦。
Version: 1.0.1
Author: TagLite
Update URI: https://cdn.taglite.com.tw/wp-plugins/taglite-recommend
*/

if (!defined('ABSPATH')) exit;

// ===================== 後台：設定頁 =====================
add_action('admin_menu', function() {
    add_options_page('TagLite 推薦設定', 'TagLite 推薦', 'manage_options', 'taglite-recommend', 'taglite_recommend_settings_page');
});

add_action('admin_init', function() {
    foreach (['title','container_id','default_thumb','type','custom_style','post_count','item_template'] as $f) {
        register_setting('taglite_recommend_options', "taglite_recommend_$f");
    }
});

function taglite_recommend_settings_page() {
    ?>
    <div class="wrap">
        <h1>TagLite 推薦文章設定</h1>
        <form method="post" action="options.php">
            <?php settings_fields('taglite_recommend_options'); ?>
            <?php do_settings_sections('taglite_recommend_options'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">推薦區標題</th>
                    <td><input type="text" name="taglite_recommend_title" value="<?php echo esc_attr(get_option('taglite_recommend_title', '推薦文章')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">推薦容器 ID</th>
                    <td><input type="text" name="taglite_recommend_container_id" value="<?php echo esc_attr(get_option('taglite_recommend_container_id', 'taglite-recommend')); ?>" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">預設圖片網址（無縮圖時使用）</th>
                    <td>
                        <input type="text" name="taglite_recommend_default_thumb" value="<?php echo esc_attr(get_option('taglite_recommend_default_thumb', '')); ?>" size="60" /><br>
                        留空代表無圖時不顯示圖片。
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">推薦類型</th>
                    <td>
                        <?php $type = get_option('taglite_recommend_type', 'post'); ?>
                        <label><input type="radio" name="taglite_recommend_type" value="post" <?php checked($type, 'post'); ?>> 文章</label><br>
                        <label><input type="radio" name="taglite_recommend_type" value="product" <?php checked($type, 'product'); ?>> 商品（WooCommerce）</label>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">自訂樣式（CSS）</th>
                    <td><textarea name="taglite_recommend_custom_style" rows="5" cols="60"><?php echo esc_textarea(get_option('taglite_recommend_custom_style', '#taglite-recommend ul {
  display:flex; flex-wrap:wrap; gap:16px; padding-left:0; margin:0; list-style:none;
}
#taglite-recommend li {
  flex:1 1 calc(33.333% - 16px); box-sizing:border-box; background:#f9f9f9; border-radius:8px; padding:12px; text-align:center; box-shadow:0 2px 6px rgba(0,0,0,0.1);
}
#taglite-recommend li a {
  display:block; text-decoration:none; color:#333; font-size:.95rem; line-height:1.4;
}
#taglite-recommend li img {
  display:block; margin:0 auto 8px; width:100%; height:auto; max-height:120px; object-fit:cover; border-radius:4px;
}
@media (max-width:768px){ #taglite-recommend li { flex:1 1 calc(50% - 16px);} }
@media (max-width:480px){ #taglite-recommend li { flex:1 1 100%; } }')); ?></textarea>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">推薦文章數量</th>
                    <td><input type="number" name="taglite_recommend_post_count" value="<?php echo esc_attr(get_option('taglite_recommend_post_count', 9)); ?>" min="1" max="20" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">推薦文章內部模板</th>
                    <td>
                        <textarea name="taglite_recommend_item_template" rows="5" cols="80"><?php echo esc_textarea(get_option('taglite_recommend_item_template', '<a href="{{url}}" target="_blank"><img src="{{thumbnail}}" alt="{{title}}">{{title}}</a>')); ?></textarea><br>
                        使用變數：<code>{{id}}</code>、<code>{{title}}</code>、<code>{{url}}</code>、<code>{{thumbnail}}</code>、<code>{{price}}</code>、<code>{{price_html}}</code><br>
                        固定外層：<code>&lt;li class="article_{{id}}"&gt;...&lt;/li&gt;</code> 會自動加上，模板中不要再包 &lt;li&gt;。
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

// ===================== 前台：樣式（head） =====================
add_action('wp_enqueue_scripts', function () {
    if (is_admin()) return;

    $containerId = esc_attr(get_option('taglite_recommend_container_id', 'taglite-recommend'));
    $customStyle = (string) get_option('taglite_recommend_custom_style', '');

    $baseCss = "#{$containerId} ul{padding-left:1.2em;list-style:disc;}"
             . "#{$containerId} li{margin-bottom:0.5em;}"
             . "#{$containerId} h4{margin-bottom:0.3em;font-size:1.2em;}";

    wp_register_style('taglite-recommend-inline', false);
    wp_enqueue_style('taglite-recommend-inline');
    wp_add_inline_style('taglite-recommend-inline', $baseCss . "\n" . $customStyle);
}, 1);

// ===================== 前台：腳本（head，順序：core → get-info → runtime） =====================
add_action('wp_enqueue_scripts', function () {
    if (is_admin()) return;

    // 若 taglite-messaging 已註冊並入列 core，這裡只需處理 get-info
    // 若未註冊，這裡會自行註冊 core（head & defer）並讓 get-info 依賴它
    if (!wp_script_is('taglite-core', 'registered')) {
        wp_register_script('taglite-core', 'https://cdn.taglite.com.tw/js/core.js', [], '1', false); // false=head
    }
    if (!wp_script_is('taglite-getinfo', 'registered')) {
        wp_register_script('taglite-getinfo', 'https://cdn.taglite.com.tw/js/get-info.js', ['taglite-core'], '1', false);
        if (function_exists('wp_script_add_data')) wp_script_add_data('taglite-getinfo', 'strategy', 'defer');
    }
    wp_enqueue_script('taglite-getinfo'); // 會自動帶上 core（若尚未入列）

    // Inline runtime（緊接 get-info.js 之後執行，保證監聽器能及早掛上）
    $runtime = [
        'TITLE'        => get_option('taglite_recommend_title', '推薦文章'),
        'CONTAINER_ID' => get_option('taglite_recommend_container_id', 'taglite-recommend'),
        'DEFAULT_THUMB'=> get_option('taglite_recommend_default_thumb', ''),
        'POST_COUNT'   => intval(get_option('taglite_recommend_post_count', 9)),
        'TPL'          => get_option('taglite_recommend_item_template') ?: '<a href="{{url}}" target="_blank"><img src="{{thumbnail}}" alt="{{title}}">{{title}}</a>',
    ];
    $settings = wp_json_encode($runtime);

    $js = <<<JS
(function(){
  const S = $settings;
  const TITLE = S.TITLE;
  const CONTAINER_ID = S.CONTAINER_ID;
  const DEFAULT_THUMB = S.DEFAULT_THUMB;
  const POST_COUNT = S.POST_COUNT;
  const TPL = S.TPL;

  function ensureContainer(cb){
    const run = () => {
      const el = document.getElementById(CONTAINER_ID);
      if (el) cb(el); else document.addEventListener('DOMContentLoaded', () => {
        const e2 = document.getElementById(CONTAINER_ID);
        if (e2) cb(e2);
      });
    };
    if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', run);
    else run();
  }

  function norm(list){
    const out = [];
    if (!list) return out;
    if (Array.isArray(list)) {
      for (const x of list) {
        if (typeof x === 'string') out.push({ tag: x, score: 1 });
        else if (x && typeof x.tag === 'string') out.push({ tag: x.tag, score: Number(x.score||0) });
      }
    } else if (typeof list === 'string') out.push({ tag: list, score: 1 });
    return out;
  }

  function getCandidates(data){
    const site   = norm(data?.onsite_tags?.tag).sort((a,b)=>(b.score||0)-(a.score||0));
    const custom = norm(data?.onsite_tags?.custom_tag).sort((a,b)=>(b.score||0)-(a.score||0));
    const seen = new Set(), out=[];
    for (const t of site)   { if (t?.tag && !seen.has(t.tag)) { seen.add(t.tag); out.push(t); } }
    for (const t of custom) { if (t?.tag && !seen.has(t.tag)) { seen.add(t.tag); out.push(t); } }
    return out;
  }

  function pick(arr,n){
    const a = arr.slice();
    for (let i=a.length-1;i>0;i--){ const j=Math.floor(Math.random()*(i+1)); [a[i],a[j]]=[a[j],a[i]]; }
    return a.slice(0,n);
  }

  function renderWithTags(userTags){
    const candidates = getCandidates(userTags);
    console.log('[TagLite 推薦文章插件] userTags 原始資料:', userTags);
    const picked = pick(candidates.slice(0,10), 3).map(t=>t.tag);
    console.log('[TagLite 推薦文章插件] 抽中的標籤:', picked);
    let apiUrl = '/wp-json/taglite/v1/recommend?count=' + POST_COUNT;
    if (picked.length) apiUrl += '&s=' + encodeURIComponent(picked.join('|'));
    console.log('[TagLite 推薦文章插件] API URL:', apiUrl);

    ensureContainer(container => {
      fetch(apiUrl).then(r=>r.json()).then(list=>{
        console.log('[TagLite 推薦文章插件] API 回傳結果:', list);
        if (!Array.isArray(list) || !list.length) return;
        const html = list.map(a =>
          '<li class="article_'+a.id+'">' + TPL
            .replace(/{{id}}/g, a.id)
            .replace(/{{title}}/g, a.title)
            .replace(/{{url}}/g, a.url)
            .replace(/{{thumbnail}}/g, a.thumbnail || DEFAULT_THUMB || '')
            .replace(/{{price}}/g, a.price ?? '')
            .replace(/{{price_html}}/g, a.price_html ?? '') +
          '</li>'
        ).join('');
        container.innerHTML = '<h4>'+TITLE+'</h4><ul>'+html+'</ul>';
      }).catch(e=>console.error('[TagLite 推薦文章插件] AJAX 錯誤', e));
    });
  }

  // 監聽事件（不要包 DOMContentLoaded，避免錯過早到的事件）
  window.addEventListener('TagLiteInfos', e => renderWithTags(e.detail));
  // 若資料已存在（可能 get-info 已經先寫入 window.TagLiteData 了），主動補觸發一次
  if (window.TagLiteData) {
    window.dispatchEvent(new CustomEvent('TagLiteInfos', { detail: window.TagLiteData }));
  }
})();
JS;

    wp_add_inline_script('taglite-getinfo', $js, 'after');
}, 2);

// ===================== Shortcode =====================
add_shortcode('taglite_recommend', function() {
    $id = esc_attr(get_option('taglite_recommend_container_id', 'taglite-recommend'));
    return "<div id='{$id}'>正在載入推薦內容...</div>";
});

// ===================== REST API =====================
add_action('rest_api_init', function () {
    register_rest_route('taglite/v1', '/recommend', [
        'methods'  => 'GET',
        'callback' => 'taglite_recommend_api',
        'permission_callback' => '__return_true'
    ]);
});

function taglite_recommend_api($request) {
    $raw_tag = trim(sanitize_text_field($request->get_param('s') ?? ''));
    $tags = array_values(array_filter(array_map('trim', explode('|', $raw_tag)), function($t){ return mb_strlen($t) > 0; }));
    $count = max(1, min(20, intval($request->get_param('count') ?? 9)));
    $type  = get_option('taglite_recommend_type', 'post');

    $results = [];
    $ids     = [];

    $base = [
        'post_type'           => $type,
        'posts_per_page'      => $count * 3,
        'post_status'         => 'publish',
        'ignore_sticky_posts' => true,
        'no_found_rows'       => true,
    ];

    $all_taxes = get_object_taxonomies($type, 'names');
    if (!is_array($all_taxes)) $all_taxes = [];

    // A) 標題 LIKE
    if (!empty($tags)) {
        $title_where = function($where) use ($tags) {
            global $wpdb;
            $likes = [];
            foreach ($tags as $kw) {
                $kw = esc_sql($wpdb->esc_like($kw));
                $likes[] = "{$wpdb->posts}.post_title LIKE '%{$kw}%'";
            }
            if ($likes) $where .= " AND (" . implode(" OR ", $likes) . ")";
            return $where;
        };
        add_filter('posts_where', $title_where, 10, 1);
        $q1 = new WP_Query($base);
        remove_filter('posts_where', $title_where, 10);

        foreach ($q1->posts ?? [] as $p) {
            $results[] = taglite_format_post_result($p, $type);
            $ids[] = $p->ID;
            if (count($results) >= $count) break;
        }
    }

    // B) taxonomy OR
    if (count($results) < $count && !empty($tags) && !empty($all_taxes)) {
        $tax_query = ['relation' => 'OR'];
        foreach ($all_taxes as $tax) {
            $tax_query[] = [
                'taxonomy' => $tax,
                'field'    => 'name',
                'terms'    => $tags,
            ];
        }
        $args2 = $base + [
            'post__not_in' => $ids,
            'orderby'      => 'rand',
            'tax_query'    => $tax_query,
        ];
        $q2 = new WP_Query($args2);
        foreach ($q2->posts ?? [] as $p) {
            $results[] = taglite_format_post_result($p, $type);
            $ids[] = $p->ID;
            if (count($results) >= $count) break;
        }
    }

    // C) 隨機保底
    if (count($results) < $count) {
        $args3 = $base + [
            'post__not_in' => $ids,
            'orderby'      => 'rand',
        ];
        $q3 = new WP_Query($args3);
        foreach ($q3->posts ?? [] as $p) {
            $results[] = taglite_format_post_result($p, $type);
            if (count($results) >= $count) break;
        }
    }

    return array_slice($results, 0, $count);
}

function taglite_format_post_result($post, $type = 'post') {
    $item = [
        'id'        => $post->ID,
        'title'     => get_the_title($post),
        'url'       => get_permalink($post),
        'thumbnail' => get_the_post_thumbnail_url($post->ID, 'medium'),
    ];
    if ($type === 'product' && function_exists('wc_get_product')) {
        $product = wc_get_product($post->ID);
        if ($product) {
            $item['price']      = $product->get_price();
            $item['price_html'] = $product->get_price_html();
        }
    }
    return $item;
}

// ===================== Plugin Update Checker（可留） =====================
$__puc_bootstrap = __DIR__ . '/inc/plugin-update-checker/plugin-update-checker.php';
if (is_readable($__puc_bootstrap)) {
    require_once $__puc_bootstrap;
    $taglite_update_checker = \YahnisElsts\PluginUpdateChecker\v5\PucFactory::buildUpdateChecker(
        'https://cdn.taglite.com.tw/wp-plugins/taglite-recommend.json',
        __FILE__,
        'taglite-recommend'
    );
    if (defined('WP_DEBUG') && WP_DEBUG && method_exists($taglite_update_checker, 'setDebugMode')) {
        $taglite_update_checker->setDebugMode(true);
    }
} else {
    error_log('[TagLite 推薦文章插件] plugin-update-checker not found: ' . $__puc_bootstrap);
}
